/************************************************************************
     File:        TargaImage.cpp

     Author:     
                  Steven J. Chenney, schenney@cs.wisc.edu
     Modifier
                  Yu-Chi Lai, yu-chi@cs.wisc.edu
     
     Comment:     Class to handle the targa image
   
     Platform:    Visio Studio.Net 2003
*************************************************************************/
#include "TargaImage.h"
#include "libtarga.h"

#include <stdlib.h>
#include <math.h>
#include <iostream>

using namespace std;

// constants
const int           RED             = 0;                // red channel
const int           GREEN           = 1;                // green channel
const int           BLUE            = 2;                // blue channel
const unsigned char BACKGROUND[3]   = { 0, 0, 0 };      // background color
//*****************************************************************************
//
// * Constructor.  Initialize member variables.
//=============================================================================
TargaImage::TargaImage(void) : width(0), height(0), data(NULL)
//=============================================================================
{}// TargaImage


//*****************************************************************************
//
// * Constructor.  Initialize member variables to values given.
//=============================================================================
TargaImage::TargaImage(int w, int h, unsigned char *d)
//=============================================================================
{
  int i;

  width = w;
  height = h;
  data = new unsigned char[width * height * 4];

  for (i = 0; i < width * height * 4; i++)
	  data[i] = d[i];
}// TargaImage


//*****************************************************************************
//
// * Destructor.  Free image memory.
//=============================================================================
TargaImage::~TargaImage()
//=============================================================================
{
  if (data)
      delete[] data;
}// ~TargaImage


//*****************************************************************************
//
// * Converts an image to RGB form, and returns the rgb pixel data - 24 
//   bits per pixel. The returned space should be deleted when no longer 
//   required.
//=============================================================================
unsigned char* TargaImage::To_RGB(void)
//=============================================================================
{
  unsigned char   *rgb = new unsigned char[width * height * 3];
  int		    i, j;

  if (! data)
	  return NULL;

  // Divide out the alpha
  for (i = 0 ; i < height ; i++) {
	  int in_offset = i * width * 4;
	  int out_offset = i * width * 3;

	  for (j = 0 ; j < width ; j++)  {
	    RGBA_To_RGB(data + (in_offset + j*4), rgb + (out_offset + j*3));
	  }
  }

  return rgb;
}// TargaImage


//*****************************************************************************
//
// * Save the image to a targa file. Returns 1 on success, 0 on failure.
//=============================================================================
bool TargaImage::Save_Image(char *filename)
//=============================================================================
{
  TargaImage	*out_image = Reverse_Rows();

  if (! out_image)
	  return false;

  if (!tga_write_raw(filename, width, height, out_image->data, TGA_TRUECOLOR_32)) {
	  cout << "TGA Save Error: %s\n", tga_error_string(tga_get_last_error());
	  return false;
  }

  delete out_image;

  return true;
}// Save_Image


//*****************************************************************************
//
// * Load a targa image from a file.  Return a new TargaImage object which 
//   must be deleted by caller.  Return NULL on failure.
//=============================================================================
TargaImage* TargaImage::Load_Image(char *filename)
//=============================================================================
{
  unsigned char   *temp_data;
  TargaImage	    *temp_image;
  TargaImage	    *result;
  int		           width, height;

  if (!filename) {
    cout << "No filename given." << endl;
    return NULL;
  }// if

  temp_data = (unsigned char*)tga_load(filename, &width, &height, TGA_TRUECOLOR_32);
  if (!temp_data) {
    cout << "TGA Error: %s\n", tga_error_string(tga_get_last_error());
	  width = height = 0;
	  return NULL;
  }

	temp_image = new TargaImage(width, height, temp_data);
  free(temp_data);

  result = temp_image->Reverse_Rows();

  delete temp_image;

  return result;
}// Load_Image


//*****************************************************************************
//
// * Given a single RGBA pixel return the single RGB equivalent composited 
//   with a black background via the second argument. 
//=============================================================================
void TargaImage::RGBA_To_RGB(unsigned char *data, unsigned char *rgb)
//=============================================================================
{
  const unsigned char	BACKGROUND[3] = { 0, 0, 0 };

  unsigned char  alpha = data[3];

  if (alpha == 0) {
    rgb[0] = BACKGROUND[0];
    rgb[1] = BACKGROUND[1];
    rgb[2] = BACKGROUND[2];
  }
  else {
	  float	alpha_scale = (float)255 / (float)alpha;
	  int	val;
	  int	i;

	  for (i = 0 ; i < 3 ; i++)  {
	    val = (int)floor(data[i] * alpha_scale);
	      if (val < 0)
					rgb[i] = 0;
	      else if (val > 255)
					rgb[i] = 255;
	      else
					rgb[i] = val;
	  }
  }
}// RGA_To_RGB


//*****************************************************************************
//
// * Copy this into a new image, reversing the rows as it goes. A pointer
//   to the new image is returned.
//=============================================================================
TargaImage* TargaImage::Reverse_Rows(void)
//=============================================================================
{
  unsigned char   *dest = new unsigned char[width * height * 4];
  TargaImage	    *result;
  int 	        i, j;

  if (! data)
    return NULL;

  for (i = 0 ; i < height ; i++) {
		int in_offset = (height - i - 1) * width * 4;
		int out_offset = i * width * 4;

		for (j = 0 ; j < width ; j++) {
			dest[out_offset + j * 4] = data[in_offset + j * 4];
			dest[out_offset + j * 4 + 1] = data[in_offset + j * 4 + 1];
			dest[out_offset + j * 4 + 2] = data[in_offset + j * 4 + 2];
			dest[out_offset + j * 4 + 3] = data[in_offset + j * 4 + 3];
		}
  }

  result = new TargaImage(width, height, dest);
  delete[] dest;
  return result;
}// Reverse_Rows




//*****************************************************************************
//
// * Transform image in RGBA format to RGB
//============================================================================
void RGBA_To_RGB(unsigned char *data, unsigned char *rgb)
//============================================================================
{
  unsigned char  alpha = data[3];

  if (alpha == 0) {
    rgb[0] = BACKGROUND[0];
    rgb[1] = BACKGROUND[1];
    rgb[2] = BACKGROUND[2];
  }
  else {
    float      alpha_scale = (float)255 / (float)alpha;
    int        val;

    for (int i = 0 ; i < 3 ; i++)    {
      val = (int)floor(data[i] * alpha_scale);
      if (val < 0)
        rgb[i] = 0;
      else if (val > 255)
        rgb[i] = 255;
      else
        rgb[i] = val;
    }
  }
}


//*****************************************************************************
//
// * Transform the image from RGB format to RGBA format
//=============================================================================
void RGB_To_RGBA(unsigned char *rgb, unsigned char alpha, unsigned char *data)
//=============================================================================
{
  if (alpha == 0) {
    data[0] = data[1] = data[2] = 0;
  }
  else {
    float      alpha_scale = (float)alpha / (float)255;
    int        val;
    int        i;

    for (i = 0 ; i < 3 ; i++)    {
      val = (int)floor(rgb[i] * alpha_scale);
      if (val < 0)
        data[i] = 0;
      else if (val > 255)
        data[i] = 255;
      else
        data[i] = val;
    }
    data[3] = alpha;
  }
}


